PROGRAM nanotube_structure

!*******************************************************************************
! ARNAUD Brice, HIREL Pierre
! Universite de Rennes1 - 2005
! 
! This program generates atomic positions of Carbon
! and Boron Nitride Nanotubes from calculated positions
! of atoms in a single sheet. Then it writes input data files
! for PAW and ABINIT programs.
!
!*******************************************************************************


IMPLICIT NONE
INTEGER :: cnatom, ixyz, n, m, natom, p, q, iat, beta_1, beta_2, jat, d
REAL*8 :: a, a0, c, length, length_perpendicular, length_parallel, zero
REAL*8 :: pi, vol, vol_cell, x, y, radius, alpha_1, alpha_2, phi, a_cell, dNT
REAL*8 :: a1(3), a2(3), a3(3), a_perpendicular(3), a_parallel(3), b1(3)
REAL*8 :: at_prim(3,2), at1(3), at2(3), at4(3), at3(3), diff(3)
REAL*8 :: b_perpendicular(3), b_parallel(3), coord(3)
INTEGER, DIMENSION(:), ALLOCATABLE :: at_type
REAL*8, DIMENSION(:,:), ALLOCATABLE :: at_pos
LOGICAL :: new, graphite
CHARACTER(LEN=3) :: c_3
CHARACTER(LEN=10) :: atom_name
CHARACTER(LEN=3) :: gra, st

pi=4.d0*DATAN(1.d0)
zero=0.d0
length=0.529177249d0   ! Bohr-Angstrm conversion factor

PRINT*, '**********************************************************'
PRINT*, '*  NANOSCAL v.0.5b                                       *'
PRINT*, '*                                                        *'
PRINT*, '*  NANOtube Structure CALculation                        *'
PRINT*, '*                                                        *'
PRINT*, '*  ARNAUD Brice, HIREL Pierre                            *'
PRINT*, '*  Universite de Rennes1 - 2005                          *'
PRINT*, '**********************************************************'
PRINT*, '*  This program generates atomic positions for any       *'
PRINT*, '*  Carbon or Boron Nitride Nanotubes by rolling          *'
PRINT*, '*  a single sheet.                                       *'
PRINT*, '**********************************************************'
PRINT*, ' '
PRINT*, 'Do you want to study a Carbon or Boron Nitride nanotube (C/BN) ?'
READ*, gra
IF(gra .EQ. 'C') THEN
   graphite = .TRUE.
   PRINT*, 'This will calculate atomic positions for Carbon Nanotubes.'
   PRINT*,'Enter the C-C bond length a0 in Bohr'
   PRINT*,'(0=experimental : a0 = 2.683 Bohr for graphene sheet)'
 ELSE
   IF(gra .EQ. 'BN') THEN
      graphite = .FALSE.
      PRINT*, 'This will calcuate atomic positions for Boron Nitride Nanotubes'
      PRINT*,'Enter the B-N bond length a0 in Bohr'
      PRINT*,'(0=experimental : a0 = 2.732 Bohr  for BN sheet)'
   ELSE
      PRINT*, 'Incorrect input data'
      STOP
   ENDIF
ENDIF

READ*, a0
IF(a0==0) THEN
   IF(graphite) THEN
      a0 = 1.42d0    ! Angstrom
   ELSE
      a0 = 1.44568d0 ! Angstrom
   ENDIF
ENDIF

a=a0*sqrt(3.d0)
WRITE(*,10),'The length of lattice vector is ', a/length, ' Bohr.'
 10 format (a32,f21.17,a7)

PRINT*,'Enter n'
READ*, n
PRINT*,'Enter m'
READ*, m


! Definition of lattice vectors (all in Angstrom)
a1(:)=0.d0
a1(1)=DSQRT(3.d0)*a/2.d0
a1(2)=a/2.d0

a2(:)=0.d0
a2(1)=DSQRT(3.d0)*a/2.d0
a2(2)=-a/2.d0

a3(:)=0.d0
a3(3)=1.d0  ! Defined for vector products, precise value not necessary

at_prim(:,:)=0.d0
at_prim(:,1)=0.d0
at_prim(1,2)=a0

CALL vect_product(a2, a3, b1)
vol_cell=DABS(DOT_PRODUCT(a1, b1))

IF (n==m) THEN
   WRITE(*,20),'The nanotube (', n, ',', m, ') is of armchair type.'
     20 format (a14,i2,a1,i2,a22)
ELSE
   IF (n==0) THEN
      WRITE(*,30),'The nanotube (', n, ',', m, ') is of zigzag type.'
       30 format (a14,i2,a1,i2,a20)
   ELSE
      IF (m==0) THEN
         m=n
         n=0
         WRITE(*,35),'The nanotube (', m, ',', n, ') is of zigzag type.'
          35 format (a14,i2,a1,i2,a20)
      ELSE
         WRITE(*,40),'The nanotube (', n, ',', m, ') is chiral.'
       40 format (a14,i2,a1,i2,a12)
      END IF
   END IF
END IF

! Calculating coordinates of nanotube basis vectors (real and reciprocal)
a_perpendicular(:)=REAL(n)*a1(:)+REAL(m)*a2(:)
d=pgcd(2*n+m, 2*m+n)
a_parallel(:)=((2.d0*REAL(m)+REAL(n))/REAL(d))*a1(:)-                  &
             &((2.d0*REAL(n)+REAL(m))/REAL(d))*a2(:)

length_perpendicular=DSQRT(DOT_PRODUCT(a_perpendicular, a_perpendicular))
length_parallel=DSQRT(DOT_PRODUCT(a_parallel, a_parallel))
radius=length_perpendicular/(2.d0*pi)                   ! Angstrom
CALL vect_product(a_parallel, a3, b_perpendicular)
vol=DABS(DOT_PRODUCT(a_perpendicular, b_perpendicular)) ! Angstrom^3
CALL vect_product(a3, a_perpendicular, b_parallel)

b_perpendicular(:)=2.d0*pi*b_perpendicular(:)/vol
b_parallel(:)=2.d0*pi*b_parallel(:)/vol

cnatom = IDNINT(2.d0*(vol/vol_cell))
WRITE(*,50), 'Theoretical number of atoms is equal to ', cnatom
 50 format (a40,i4)

ALLOCATE(at_pos(3, cnatom))
ALLOCATE(at_type(cnatom))
at_pos(:,:)=0.d0
at_type(:)=0

natom=0
DO p=-100, 100
   DO q=-100, 100
      DO iat=1, 2
         coord(:)=at_prim(:, iat)+REAL(p)*a1(:)+REAL(q)*a2(:)  
         x=DOT_PRODUCT(coord, a_perpendicular)/length_perpendicular
         y=DOT_PRODUCT(coord, a_parallel)/length_parallel
         IF ( ((x>=0.d0).AND.(x<=length_perpendicular)).AND.         &
            &((y>=0.d0).AND.(y<=length_parallel))) THEN
            new=.TRUE.
            DO jat=1, natom
               diff(:)=coord(:)-at_pos(:, jat)
               alpha_1=DOT_PRODUCT(diff, b_perpendicular)/(2.d0*pi)
               alpha_2=DOT_PRODUCT(diff, b_parallel)/(2.d0*pi)
               beta_1=IDNINT(alpha_1)
               beta_2=IDNINT(alpha_2)
               IF ((DABS(alpha_1-beta_1)<1.d-6).AND.                &
                  &(DABS(alpha_2-beta_2)<1.d-6)) THEN
                  new=.FALSE.
                  EXIT
               END IF
            END DO
            IF (new) THEN
               natom= natom+1
               at_pos(:, natom)=coord(:)
               at_type(natom)=iat
            END IF
         END IF
      END DO ! end loop over iat
   END DO ! end loop over q
END DO !end loop over p

WRITE(*,55),'Calculated number of atoms is ', natom
 55 format (a34,i4)

! Check if theoretical and calculated number of atoms match
IF(natom .NE. cnatom) THEN
   PRINT*, ' '
   PRINT*, 'WARNING : There is a disaccordance between theoretical and calculated '
   PRINT*, 'number of atoms. Atomic positions or other results may not be accurate.'
   PRINT*, 'Continue anyway ? (y/n)'
   READ*, st
   IF(st .NE. 'y') THEN
      IF(st .NE. 'n') THEN
         PRINT*, 'Incorrect answer : continue anyway ? (y/n)'
         READ*, st
         IF(st .NE. 'y') THEN
            st='n'
         ENDIF
      ELSE
         PRINT*, 'Program exit - no file generated.'
         STOP
      ENDIF
   ENDIF
ENDIF

WRITE(*, 60),'Diamater of non-relaxed structure is', 2.d0*radius/length, ' Bohr.'
 60 format (a40,f21.17,a11)
WRITE(*, 70) 'Volume of nanotube unit cell is ', vol/(length**3.d0), ' Bohr^3.'
 70 format (a32,f25.17,a8)

! Writing characteristics in a file
IF (graphite) THEN
   OPEN(UNIT=100,FILE='char_CNT',FORM='FORMATTED',STATUS='UNKNOWN')
   WRITE(100,*) 'Characteristics of Carbon nanotube (just as an indication)'
   OPEN(unit=300, FILE='C_NT_paw_strc',FORM='FORMATTED', STATUS='UNKNOWN')
   OPEN(unit=400, FILE='C_NT_abinit_BS_in',FORM='FORMATTED', STATUS='UNKNOWN')
   OPEN(unit=500, FILE='C_NT_abinit_relax_in',FORM='FORMATTED', STATUS='UNKNOWN')
ELSE
   OPEN(UNIT=100,FILE='char_BNNT',FORM='FORMATTED',STATUS='UNKNOWN')
   WRITE(100,*) 'Characteristics of Boron Nitride nanotube (just as an indication)'
   OPEN(unit=300, FILE='BN_NT_paw_strc',FORM='FORMATTED', STATUS='UNKNOWN')
   OPEN(unit=400, FILE='BN_NT_abinit_BS_in',FORM='FORMATTED', STATUS='UNKNOWN')
   OPEN(unit=500, FILE='BN_NT_abinit_relax_in',FORM='FORMATTED', STATUS='UNKNOWN')
END IF

WRITE(100,*), '=============================================================='
WRITE(100,*), ' '
WRITE(100,*), 'Basis vectors for sheet (Bohr) :'
WRITE(100,*), ' a1 = ', a1(1)/length, a1(2)/length, a1(3)/length
WRITE(100,*), ' a2 = ', a2(1)/length, a2(2)/length, a2(3)/length
WRITE(100,*), 'Basis vectors for sheet (Angstrom) :'
WRITE(100,*), ' a1 = ', a1(1), a1(2), a1(3)
WRITE(100,*), ' a2 = ', a2(1), a2(2), a2(3)
WRITE(100,*), ' '
WRITE(100,72), 'Coordinates of the nanotube in (a1,a2) basis : (', n, ',', m, ')'
 72 format (a48,i3,a1,i3,a1)
WRITE(100,*), ' '
WRITE(100,*), 'Number of atoms = ', natom
WRITE(100,*), 'Diameter (Bohr) = ', 2.d0*radius/length
WRITE(100,*), 'Diameter (Angstroms) = ', 2.d0*radius
WRITE(100,*), 'Volume of nanotube unit cell (Bohr^3) : ', vol/(length**3.d0)
WRITE(100,*), 'Volume of nanotube unit cell (A^3) : ', vol
IF (graphite) THEN
   PRINT*, '-> Wrote nanotube characteristics in file  char_CNT.'
ELSE
   PRINT*, '-> Wrote nanotube characteristics in file  char_BNNT.'
END IF
CLOSE(UNIT=100)

PRINT*, ' ' 
PRINT*, 'Nanotubes will be arranged in an hexagonal fashion.'
PRINT*, 'Enter distance between two nanotubes shells (in A) :'
IF(graphite) THEN
   PRINT*, '(0=distance between two sheets in graphite structure, 6.338 Bohr)'
ELSE
   PRINT*, '(0=distance between two sheets in h-BN structure, 6.2909 Bohr)'
ENDIF
READ*, dNT
IF(dNT==0) THEN
   IF(graphite) THEN
      dNT=3.35395d0
   ELSE
      dNT=3.3290011557341d0
   ENDIF
ENDIF
IF(dNT<0) THEN
   PRINT*, 'Incorrect distance. Program will now exit.'
ENDIF
a_cell = 2.d0*radius + dNT


IF (graphite) THEN
   OPEN(unit=600,FILE='C_sheet.xsf',FORM='FORMATTED',STATUS='UNKNOWN')
   WRITE(600,*)"ATOMS"
   DO iat=1, natom
!======================================================================
!     Carbon atom
!======================================================================
      WRITE(600,*)"6 ", (at_pos(ixyz, iat), ixyz=1, 3)
   END DO
   CLOSE(unit=600)
ELSE
   OPEN(unit=600,FILE='BN_sheet.xsf',FORM='FORMATTED',STATUS='UNKNOWN')
   WRITE(600,*)"ATOMS"
   DO iat=1, natom
      IF (at_type(iat)==1) THEN
!======================================================================
!        Boron atom
!======================================================================
         WRITE(600,*)"5 ", (at_pos(ixyz, iat), ixyz=1, 3)
      ELSE
!======================================================================
!        Nitrogen atom
!======================================================================
         WRITE(600,*)"7 ", (at_pos(ixyz, iat), ixyz=1, 3)
      END IF
   END DO
   CLOSE(unit=600)
END IF

IF (graphite) THEN
   OPEN(unit=600,FILE='C_NT_atoms.xsf',FORM='FORMATTED',STATUS='UNKNOWN')
   OPEN(unit=700,FILE='C_NT_structure.xsf',FORM='FORMATTED',STATUS='UNKNOWN')
ELSE
   OPEN(unit=600,FILE='BN_NT_atoms.xsf',FORM='FORMATTED',STATUS='UNKNOWN')
   OPEN(unit=700,FILE='BN_NT_structure.xsf',FORM='FORMATTED',STATUS='UNKNOWN')
END IF

WRITE(600,*) "ATOMS"
WRITE(700,*) "CRYSTAL"
WRITE(700,*) "PRIMVEC"               ! Input in Bohr in XCrysDen
WRITE(700,*) a_cell, zero, zero
WRITE(700,*) a_cell/2.d0, a_cell*DSQRT(3.d0)/2.d0, zero
WRITE(700,*) zero, zero, length_parallel
WRITE(700,*) "CONVVEC"
WRITE(700,*) a_cell, zero, zero
WRITE(700,*) a_cell/2.d0, a_cell*DSQRT(3.d0)/2.d0, zero
WRITE(700,*) zero, zero, length_parallel
WRITE(700,*) "PRIMCOORD"
WRITE(700,*) natom, " 1"
DO iat=1, natom
   coord(:)=at_pos(:, iat)
   x=DOT_PRODUCT(coord, a_perpendicular)/length_perpendicular
   phi=x/radius
   y=DOT_PRODUCT(coord, a_parallel)/length_parallel
   IF (graphite) THEN
      WRITE(600,*) "6 ", radius*DCOS(phi), radius*DSIN(phi), y
      WRITE(700,*) "6 ", radius*DCOS(phi)+(3*a_cell/4.d0),               &
                  &      radius*DSIN(phi)+(a_cell*DSQRT(3.d0)/4.d0), y
   ELSE
      IF (at_type(iat)==1) THEN 
         WRITE(600,*) "5 ", radius*DCOS(phi), radius*DSIN(phi), y
         WRITE(700,*) "5 ", radius*DCOS(phi)+(3*a_cell/4.d0),            &
                  &         radius*DSIN(phi)+(a_cell*DSQRT(3.d0)/4.d0), y
      ELSE
         WRITE(600,*) "7 ", radius*DCOS(phi), radius*DSIN(phi), y
         WRITE(700,*) "7 ", radius*DCOS(phi)+(3*a_cell/4.d0),            &
                  &         radius*DSIN(phi)+(a_cell*DSQRT(3.d0)/4.d0), y
      END IF
   END IF
END DO
CLOSE(unit=600)
CLOSE(unit=700)


WRITE(300,*)" !STRUCTURE"                         ! Input in Angstrom in PAW
WRITE(300,*)" !GENERIC LUNIT=1.000000 !END"
WRITE(300,75) ' !OCCUPATIONS NBAND=', INT((natom/2.d0)*8.d0), '!END'
 75 format (a19,i3,a5)
WRITE(300,*) ' !KPOINTS DIV= 2 2 8 !END'
WRITE(300,80) " !LATTICE T= ", a_cell/length, zero, zero
WRITE(300,80)  '             ', a_cell/(2.d0*length), a_cell*DSQRT(3.d0)/(2.d0*length), zero
 80 format (a13,3f25.17)
WRITE(300,90) '             ', zero, zero, length_parallel/length, " !END"
 90 format (a13,3f25.17,a5)


WRITE(400,*) '# Non-relaxed nanotube band structure'
WRITE(500,*) '# Relaxation of cell vector along the nanotube axis'
WRITE(500,*) '# and of atomic positions'
WRITE(400,*) 'ndtset 2'
WRITE(400,*) '#Dataset 1 : usual self-consistent calculation'
WRITE(400,*) 'kptopt1 1'
WRITE(400,*) 'nshifk1 1'
WRITE(400,*) 'shiftk1 0.0 0.0 0.5'
WRITE(400,*) 'ngkpt1 1 1 4'
WRITE(400,*) 'prtden1 1'
WRITE(400,*) 'toldfe1 1.0d-6'
WRITE(400,*) ' '
WRITE(400,*) '#Dataset 2 : the band structure'
WRITE(400,*) 'iscf2 -2'
WRITE(400,*) 'getden2 -1'
WRITE(400,*) 'kptopt2 -2'
WRITE(400,95) ' nband2 ', INT((natom/2.d0)*8.d0)
 95 format (a8,i3)
WRITE(400,*) 'ndivk2 10 10'
WRITE(400,*) 'kptbounds2 0.0 0.0 0.0'
WRITE(400,*) '0.0 0.0 0.5'
WRITE(400,*) '0.0 0.0 1.0'
WRITE(400,*) 'tolwfr2 1.d-12'
WRITE(400,*) 'enunit2 1'
WRITE(400,*) ' '
WRITE(500,*) ' '
WRITE(400,*) '#Definition of the unit cell'
WRITE(500,*) '#Definition of the unit cell'         ! Input in Angstrm in ABINIT
WRITE(400,100) 'acell ', a_cell/length, a_cell/length, length_parallel/length
WRITE(500,100) 'acell ', a_cell/length, a_cell/length, length_parallel/length
 100 format (a7,3f25.17)
WRITE(400,*) 'rprim 1.0 0.0 0.0'                    !
WRITE(400,*) '      0.5 0.8660254037844386 0.0'     ! Primitive vectors
WRITE(400,*) '      0.0 0.0 1.0'                    !
WRITE(500,*) 'rprim 1.0 0.0 0.0'                    ! scaled by acell
WRITE(500,*) '      0.5 0.8660254037844386 0.0'     !
WRITE(500,*) '      0.0 0.0 1.0'                    !
WRITE(500,*) ' ecut  25'
WRITE(500,*) ' nkpt  20'
WRITE(500,*) ' ecutsm  0.5'
WRITE(500,*) ' dilatmx  1.3'
WRITE(500,*) ' strprecon  0.2'
WRITE(500,*) ' optcell 6'
WRITE(500,*) ' ionmov  3'
WRITE(500,*) ' ntime  40'
WRITE(500,*) ' tolmxf  5.0d-4'
WRITE(500,*) ' toldff  5.0d-5'
WRITE(500,110) ' nband ',  INT((natom/2.d0)*8.d0)
 110 format (a8,i3)
WRITE(400,*) ' '
WRITE(500,*) ' '
WRITE(400,*) '#Definition of the atom types'
WRITE(500,*) '#Definition of the atom types'

IF (graphite) THEN
      WRITE(300,*)"!SPECIES NAME='C_' ZV=3.0 !END"
      WRITE(400,*) 'ntypat 1'
      WRITE(400,*) 'znucl 6'
      WRITE(500,*) 'ntypat 1'
      WRITE(500,*) 'znucl 6'
   ELSE
      WRITE(300,120) "!SPECIES NAME='B_' ZV=3.0 M=10.811 PSEKIN=0.0 NPRO=1 1 0 &
                  &FILE='/home/stagiaire/PAW/Exec/h_BN/b_.75_6.0_dft1.out' !END"
      WRITE(300,120) "!SPECIES NAME='N_' ZV=5.0 M=14.0067 PSEKIN=0.0 NPRO=2 2 0 &
                  &FILE='/home/stagiaire/PAW/Exec/h_BN/n_.75_6.0_dft1.out' !END"
       120 format (a125)
      WRITE(400,*) 'ntypat 2'
      WRITE(400,*) 'znucl 5 7'
      WRITE(500,*) 'ntypat 2'
      WRITE(500,*) 'znucl 5 7'
ENDIF

WRITE(400,130) 'natom ', natom
WRITE(500,130) 'natom ', natom
 130 format (a6,i3)

WRITE(400,*) 'typat'
WRITE(500,*) 'typat'
DO iat=1,natom
   IF (graphite) THEN
      WRITE(400,*) '1 '
      WRITE(500,*) '1 '
   ELSE
      IF (at_type(iat)==1) THEN
         WRITE(400,*) '1 '
         WRITE(500,*) '1 '
      ELSE
         WRITE(400,*) '2 '
         WRITE(500,*) '2 '
      ENDIF
   ENDIF
END DO

WRITE(400,*) 'xcart '
WRITE(500,*) 'xcart '

DO iat=1, natom
   coord(:)=at_pos(:, iat)
   x=DOT_PRODUCT(coord, a_perpendicular)/length_perpendicular
   phi=x/radius
   y=DOT_PRODUCT(coord, a_parallel)/length_parallel
   CALL transform_character(iat, c_3)
   IF (graphite) THEN
      atom_name="'C_"//trim(c_3)//"'"
   ELSE
      IF (at_type(iat)==1) THEN
         atom_name="'B_"//trim(c_3)//"'"
      ELSE
         atom_name="'N_"//trim(c_3)//"'"
      END IF
   END IF
   WRITE(300,140) "!ATOM NAME=", trim(atom_name),"   R= ", (radius*DCOS(phi)+(3*a_cell/4.d0))/length,&
                                & (radius*DSIN(phi)+(a_cell*DSQRT(3.d0)/4.d0))/length, y/length, " !END "
    140 format (a13,a6,a7,3f25.17,a6)
   WRITE(400,150) (radius*DCOS(phi)+(3*a_cell/4.d0))/length, &
        &         (radius*DSIN(phi)+(a_cell*DSQRT(3.d0)/4.d0))/length, y/length
   WRITE(500,150) (radius*DCOS(phi)+(3*a_cell/4.d0))/length, &
        &         (radius*DSIN(phi)+(a_cell*DSQRT(3.d0)/4.d0))/length, y/length
    150 format (3f22.17)
END DO
WRITE(300,*)"!END"
WRITE(300,*)"!EOB"
WRITE(400,*) '#Definition of the plane wave basis set'
WRITE(400,*) 'ecut 15.0  #Cutoff in Hartree=Rydberg/2'
WRITE(400,*) '#Definition of the SCF procedure'
WRITE(400,*) 'nstep 20'
WRITE(400,*) 'diemac 12.0'
WRITE(500,*) '#Definition of the SCF procedure'
WRITE(500,*) 'nstep 30'
WRITE(500,*) 'diemac 12.0'

CLOSE(UNIT=100)
CLOSE(unit=300)
CLOSE(unit=400)
CLOSE(unit=500)


IF(graphite) THEN
   PRINT*, '-> Wrote XCrySDen preview file : C_NT_structure.xsf'
   PRINT*,'-> Wrote PAW structure file : C_NT_paw_strc'
   PRINT*,'-> Wrote ABINIT input file for band structure calculation : C_NT_abinit_BS_in'
   PRINT*,'-> Wrote ABINIT input file for structure relaxation : C_NT_abinit_relax_in'
ELSE
   PRINT*, '-> Wrote XCrySDen preview file : BN_NT_structure.xsf'
   PRINT*,'-> Wrote PAW structure file : BN_NT_paw_strc'
   PRINT*,'-> Wrote ABINIT input file for band structure calculation : BN_NT_abinit_BS_in'
   PRINT*,'-> Wrote ABINIT input file for structure relaxation : BN_NT_abinit_relax_in'
ENDIF

DEALLOCATE(at_pos)
DEALLOCATE(at_type)

PRINT*, ' '
PRINT*, '*** WARNING ***'
PRINT*, 'Some parameters in written files may be inaccurate for specified nanotube.'
PRINT*, 'Please check files before performing any calculation.'
PRINT*, ' '
PRINT*, '  (`._.``._.- END OF PROGRAM -._.``._.`)'
PRINT*, ' '

CONTAINS


SUBROUTINE vect_product(a, b, c)

IMPLICIT NONE
REAL*8, INTENT(IN)  :: a(3)
REAL*8, INTENT(IN)  :: b(3)
REAL*8, INTENT(OUT) :: c(3)

c(1)=a(2)*b(3)-a(3)*b(2)
c(2)=a(3)*b(1)-a(1)*b(3)
c(3)=a(1)*b(2)-a(2)*b(1)

END SUBROUTINE vect_product

FUNCTION pgcd(l,m)
IMPLICIT NONE
INTEGER :: pgcd
INTEGER, INTENT(IN) :: l
INTEGER, INTENT(IN) :: m

INTEGER :: c, r, a, b

a=l
b=m

IF (a==b) THEN
   pgcd=a
ELSE
   IF (a<b) THEN
      c=a
      a=b
      b=c
   END IF
   r=1
   DO WHILE (r>0)
      r=MOD(a, b)
      a=b
      b=r 
   END DO
   pgcd=a
END IF


END FUNCTION pgcd 

SUBROUTINE transform_character(number, c_3)


IMPLICIT NONE
INTEGER, INTENT(IN) :: number
CHARACTER(len=3), INTENT(OUT) :: c_3

INTEGER :: unite, dizaine, centaine, number_of_character
CHARACTER (len=1) :: zero, one, two, three, four, five, six, seven, eight, nine
CHARACTER (len=1) :: temp1, temp2, temp3

zero="0"
one="1"
two="2"
three="3"
four="4"
five="5"
six="6"
seven="7"
eight="8"
nine="9"

number_of_character=1
centaine=number/100
dizaine=(number-100*centaine)/10
IF (dizaine/=0) THEN
   number_of_character=2
END IF

IF (centaine/=0) THEN
   number_of_character=3
END IF
unite=(number-100*centaine-10*dizaine)

IF (number_of_character==3) THEN
   IF (centaine==1) THEN
      temp1=one
   END IF
   IF (centaine==2) THEN
      temp1=two
   END IF

   IF (centaine==3) THEN
      temp1=three
   END IF

   IF (centaine==4) THEN
      temp1=four
   END IF

   IF (centaine==5) THEN
      temp1=five
   END IF

   IF (centaine==6) THEN
      temp1=six
   END IF

   IF (centaine==7) THEN
      temp1=seven
   END IF

   IF (centaine==8) THEN
      temp1=eight
   END IF

   IF (centaine==9) THEN
      temp1=nine
   END IF

   IF (dizaine==0) THEN
      temp2=zero
   END IF

   IF (dizaine==1) THEN
      temp2=one
   END IF
   IF (dizaine==2) THEN
      temp2=two
   END IF

   IF (dizaine==3) THEN
      temp2=three
   END IF

   IF (dizaine==4) THEN
      temp2=four
   END IF

   IF (dizaine==5) THEN
      temp2=five
   END IF

   IF (dizaine==6) THEN
      temp2=six
   END IF

   IF (dizaine==7) THEN
      temp2=seven
   END IF
   IF (dizaine==8) THEN
      temp2=eight
   END IF

   IF (dizaine==9) THEN
      temp2=nine
   END IF

   IF (unite==0) THEN
      temp3=zero
   END IF

   IF (unite==1) THEN
      temp3=one
   END IF
   IF (unite==2) THEN
      temp3=two
   END IF

   IF (unite==3) THEN
      temp3=three
   END IF

   IF (unite==4) THEN
      temp3=four
   END IF

   IF (unite==5) THEN
      temp3=five
   END IF

   IF (unite==6) THEN
      temp3=six
   END IF

   IF (unite==7) THEN
      temp3=seven
   END IF

   IF (unite==8) THEN
      temp3=eight
   END IF

   IF (unite==9) THEN
      temp3=nine
   END IF

   c_3=temp1//temp2//temp3

END IF
IF (number_of_character==2) THEN

   IF (dizaine==1) THEN
      temp2=one
   END IF
   IF (dizaine==2) THEN
      temp2=two
   END IF

   IF (dizaine==3) THEN
      temp2=three
   END IF

   IF (dizaine==4) THEN
      temp2=four
   END IF

   IF (dizaine==5) THEN
      temp2=five
   END IF

   IF (dizaine==6) THEN
      temp2=six
   END IF

   IF (dizaine==7) THEN
      temp2=seven
   END IF

   IF (dizaine==8) THEN
      temp2=eight
   END IF

   IF (dizaine==9) THEN
      temp2=nine
   END IF

   IF (unite==0) THEN
      temp3=zero
   END IF

   IF (unite==1) THEN
      temp3=one
   END IF
   IF (unite==2) THEN
      temp3=two
   END IF

   IF (unite==3) THEN
      temp3=three
   END IF

   IF (unite==4) THEN
      temp3=four
   END IF

   IF (unite==5) THEN
      temp3=five
   END IF

   IF (unite==6) THEN
      temp3=six
   END IF

   IF (unite==7) THEN
      temp3=seven
   END IF

   IF (unite==8) THEN
      temp3=eight
   END IF

   IF (unite==9) THEN
      temp3=nine
   END IF

   c_3=temp2//temp3

END IF


IF (number_of_character==1) THEN

   IF (unite==0) THEN
      temp3=zero
   END IF

   IF (unite==1) THEN
      temp3=one
   END IF
   IF (unite==2) THEN
      temp3=two
   END IF

   IF (unite==3) THEN
      temp3=three
   END IF

   IF (unite==4) THEN
      temp3=four
   END IF

   IF (unite==5) THEN
      temp3=five
   END IF

   IF (unite==6) THEN
      temp3=six
   END IF

   IF (unite==7) THEN
      temp3=seven
   END IF

   IF (unite==8) THEN
      temp3=eight
   END IF

   IF (unite==9) THEN
      temp3=nine
   END IF

   c_3=temp3

END IF

END SUBROUTINE transform_character

END PROGRAM
